/******************************************************************************
*
* Freescale Semiconductor Inc.
* (c) Copyright 2006 Freescale Semiconductor, Inc.
* ALL RIGHTS RESERVED.
*
*******************************************************************************
*
* $File Name:  libdsp2.h$
*
* $Date:       Jul-21-2006$
*
* $Version:    1.0.3.0$
*
* Description: Header file for the MPC5500 Function Library 2
*
*******************************************************************************/

#ifndef __LIBDSP2_H
#define __LIBDSP2_H


/* seed tables for 16-bit data */
extern  unsigned short seed_radix2_16bit_64[8];
extern  unsigned short seed_radix2_16bit_128[16];
#define seed_radix2_16bit_256 seed_radix2_16bit_128    /* 256 seed table is the same as 128 */
extern  unsigned short seed_radix2_16bit_512[32];
#define seed_radix2_16bit_1024 seed_radix2_16bit_512   /* 1024 seed table is the same as 512 */
extern  unsigned short seed_radix2_16bit_2048[64];
#define seed_radix2_16bit_4096 seed_radix2_16bit_2048  /* 4096 seed table is the same as 2048 */

extern  unsigned short seed_radix4_16bit_64[16];
#define seed_radix4_16bit_256 seed_radix4_16bit_64     /* 256 seed table is the same as 64 */
extern  unsigned short seed_radix4_16bit_1024[64];
#define seed_radix4_16bit_4096 seed_radix4_16bit_1024  /* 4096 seed table is the same as 1024 */

/* seed tables for 32-bit data */
extern  unsigned short seed_radix2_32bit_64[8];
extern  unsigned short seed_radix2_32bit_128[16];
#define seed_radix2_32bit_256 seed_radix2_32bit_128    /* 256 seed table is the same as 128 */
extern  unsigned short seed_radix2_32bit_512[32];
#define seed_radix2_32bit_1024 seed_radix2_32bit_512   /* 1024 seed table is the same as 512 */
extern  unsigned short seed_radix2_32bit_2048[64];
#define seed_radix2_32bit_4096 seed_radix2_32bit_2048  /* 4096 seed table is the same as 2048 */

extern  unsigned short seed_radix4_32bit_64[16];
#define seed_radix4_32bit_256 seed_radix4_32bit_64     /* 256 seed table is the same as 64 */
extern  unsigned short seed_radix4_32bit_1024[64];
#define seed_radix4_32bit_4096 seed_radix4_32bit_1024  /* 4096 seed table is the same as 1024 */


/* float and frac32 twiddle factor tables for radix-2 */
extern float w_table_radix2_float_64[64];
extern float w_table_radix2_float_128[128];
extern float w_table_radix2_float_256[256];
extern float w_table_radix2_float_512[512];
extern float w_table_radix2_float_1024[1024];
extern float w_table_radix2_float_2048[2048];
extern float w_table_radix2_float_4096[4096];
extern int   w_table_radix2_frac32_64[64];
extern int   w_table_radix2_frac32_128[128];
extern int   w_table_radix2_frac32_256[256];
extern int   w_table_radix2_frac32_512[512];
extern int   w_table_radix2_frac32_1024[1024];
extern int   w_table_radix2_frac32_2048[2048];
extern int   w_table_radix2_frac32_4096[4096];

/* float and frac32 twiddle factor tables for radix-4 */
extern float w_table_radix4_float_64[92];              /* 3*N/2-4 */
extern float w_table_radix4_float_256[380];
extern float w_table_radix4_float_1024[1532];
extern float w_table_radix4_float_4096[6140];
extern int   w_table_radix4_frac32_64[92];
extern int   w_table_radix4_frac32_256[380];
extern int   w_table_radix4_frac32_1024[1532];
extern int   w_table_radix4_frac32_4096[6140];

/* float and frac32 twiddle factor tables for split function of real to complex FFT */
extern float wa_table_float_128[32+1];                 /* N/4+1 */
extern float wa_table_float_256[64+1];
extern float wa_table_float_512[128+1];
extern float wa_table_float_1024[256+1];
extern float wa_table_float_2048[512+1];
extern float wa_table_float_4096[1024+1];
extern float wb_table_float_128[32+1];
extern float wb_table_float_256[64+1];
extern float wb_table_float_512[128+1];
extern float wb_table_float_1024[256+1];
extern float wb_table_float_2048[512+1];
extern float wb_table_float_4096[1024+1];
extern int   wa_table_frac32_128[32+1];                /* N/4+1 */
extern int   wa_table_frac32_256[64+1];
extern int   wa_table_frac32_512[128+1];
extern int   wa_table_frac32_1024[256+1];
extern int   wa_table_frac32_2048[512+1];
extern int   wa_table_frac32_4096[1024+1];
extern int   wb_table_frac32_128[32+1];
extern int   wb_table_frac32_256[64+1];
extern int   wb_table_frac32_512[128+1];
extern int   wb_table_frac32_1024[256+1];
extern int   wb_table_frac32_2048[512+1];
extern int   wb_table_frac32_4096[1024+1];



/* function prototypes */
void bitrev_table_16bit(unsigned int n, short *inout_buffer,
					unsigned short *seed_table);
void bitrev_table_32bit(unsigned int n, float *inout_buffer,
					unsigned short *seed_table);
void fft_radix4_float(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_radix4_frac32(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_quad_float(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_quad_frac32(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_radix2_last_stage_float(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_radix2_last_stage_frac32(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_real_split_float(unsigned int N, float *y, float *wa, float *wb);
void fft_real_split_frac32(unsigned int N, int *y, int *wa, int *wb);
void window_apply_complex_float(unsigned int N, float *x, float *y, float *w);
void window_apply_complex_frac16(unsigned int N, short *x, short *y, short *w);
void window_apply_real_float(unsigned int N, float *x, float *y, float *w);
void window_apply_real_frac16(unsigned int N, short *x, short *y, short *w);



/* function prototypes of C functions */
void bitrev_table_16bit_c(unsigned int n, short *inout_buffer,
					unsigned short *seed_table);
void bitrev_table_32bit_c(unsigned int n, float *inout_buffer,
					unsigned short *seed_table);
void fft_radix4_float_c(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_radix4_frac32_c(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_quad_float_c(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_quad_frac32_c(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_radix2_last_stage_float_c(unsigned int N, float *inout_buffer,
					float *twiddle_factor_table);
void fft_radix2_last_stage_frac32_c(unsigned int N, int *inout_buffer,
					int *twiddle_factor_table);
void fft_real_split_float_c(unsigned int N, float *y, float *wa, float *wb);
void window_apply_complex_float_c(unsigned int N, float *x, float *y, float *w);
void window_apply_real_float_c(unsigned int N, float *x, float *y, float *w);


/******************** radix-4 FFTs macros *********************/
/* radix-4 float FFT macros */
#define fft_radix4_float_64(inout_buffer) \
    bitrev_table_32bit(4, inout_buffer, seed_radix4_32bit_64); \
    fft_radix4_float (64, inout_buffer, w_table_radix4_float_64)

#define fft_radix4_float_256(inout_buffer) \
    bitrev_table_32bit(16, inout_buffer, seed_radix4_32bit_256); \
    fft_radix4_float (256, inout_buffer, w_table_radix4_float_256)

#define fft_radix4_float_1024(inout_buffer) \
    bitrev_table_32bit(16, inout_buffer, seed_radix4_32bit_1024); \
    fft_radix4_float(1024, inout_buffer, w_table_radix4_float_1024)

#define fft_radix4_float_4096(inout_buffer) \
    bitrev_table_32bit(64, inout_buffer, seed_radix4_32bit_4096); \
    fft_radix4_float(4096, inout_buffer, w_table_radix4_float_4096)

/* radix-4 frac16/frac32 FFT macros */
#define fft_radix4_frac32_64(inout_buffer) \
    bitrev_table_16bit(4, (short *)(inout_buffer+64), seed_radix4_16bit_64); \
    fft_radix4_frac32(64, inout_buffer, w_table_radix4_frac32_64)

#define fft_radix4_frac32_256(inout_buffer) \
    bitrev_table_16bit(16, (short *)(inout_buffer+256), seed_radix4_16bit_256); \
    fft_radix4_frac32(256, inout_buffer, w_table_radix4_frac32_256)

#define fft_radix4_frac32_1024(inout_buffer) \
    bitrev_table_16bit (16, (short *)(inout_buffer+1024), seed_radix4_16bit_1024); \
    fft_radix4_frac32(1024, inout_buffer, w_table_radix4_frac32_1024)

#define fft_radix4_frac32_4096(inout_buffer) \
    bitrev_table_16bit (64, (short *)(inout_buffer+4096), seed_radix4_16bit_4096); \
    fft_radix4_frac32(4096, inout_buffer, w_table_radix4_frac32_4096)



/******************** quad FFTs macros *********************/
/* float */
#define fft_quad_float_64(inout_buffer) \
    bitrev_table_32bit(8, inout_buffer, seed_radix2_32bit_64); \
    fft_quad_float(64, inout_buffer, w_table_radix2_float_64)

#define fft_quad_float_256(inout_buffer) \
    bitrev_table_32bit(16, inout_buffer, seed_radix2_32bit_256); \
    fft_quad_float(256, inout_buffer, w_table_radix2_float_256)

#define fft_quad_float_1024(inout_buffer) \
    bitrev_table_32bit(32, inout_buffer, seed_radix2_32bit_1024); \
    fft_quad_float(1024, inout_buffer, w_table_radix2_float_1024)

#define fft_quad_float_4096(inout_buffer) \
    bitrev_table_32bit(64, inout_buffer, seed_radix2_32bit_4096); \
    fft_quad_float(4096, inout_buffer, w_table_radix2_float_4096)


#define fft_quad_float_128(inout_buffer) \
    bitrev_table_32bit(8, inout_buffer, seed_radix2_32bit_128); \
    fft_quad_float(128, inout_buffer, w_table_radix2_float_128); \
    fft_radix2_last_stage_float(128, inout_buffer, w_table_radix2_float_128)

#define fft_quad_float_512(inout_buffer) \
    bitrev_table_32bit(16, inout_buffer, seed_radix2_32bit_512); \
    fft_quad_float(512, inout_buffer, w_table_radix2_float_512); \
    fft_radix2_last_stage_float(512, inout_buffer, w_table_radix2_float_512)

#define fft_quad_float_2048(inout_buffer) \
    bitrev_table_32bit(32, inout_buffer, seed_radix2_32bit_2048); \
    fft_quad_float(2048, inout_buffer, w_table_radix2_float_2048); \
    fft_radix2_last_stage_float(2048, inout_buffer, w_table_radix2_float_2048)




/* frac16/frac32 */
#define fft_quad_frac32_64(inout_buffer) \
    bitrev_table_16bit(8, (short *)(inout_buffer+64), seed_radix2_16bit_64); \
    fft_quad_frac32(64, inout_buffer, w_table_radix2_frac32_64)

#define fft_quad_frac32_256(inout_buffer) \
    bitrev_table_16bit(16, (short *)(inout_buffer+256), seed_radix2_16bit_256); \
    fft_quad_frac32(256, inout_buffer, w_table_radix2_frac32_256)

#define fft_quad_frac32_1024(inout_buffer) \
    bitrev_table_16bit(32, (short *)(inout_buffer+1024), seed_radix2_16bit_1024); \
    fft_quad_frac32(1024, inout_buffer, w_table_radix2_frac32_1024)

#define fft_quad_frac32_4096(inout_buffer) \
    bitrev_table_16bit(64, (short *)(inout_buffer+4096), seed_radix2_16bit_4096); \
    fft_quad_frac32(4096, inout_buffer, w_table_radix2_frac32_4096)


#define fft_quad_frac32_128(inout_buffer) \
    bitrev_table_16bit(8, (short *)(inout_buffer+128), seed_radix2_16bit_128); \
    fft_quad_frac32(128, inout_buffer, w_table_radix2_frac32_128); \
    fft_radix2_last_stage_frac32(128, inout_buffer, w_table_radix2_frac32_128)

#define fft_quad_frac32_512(inout_buffer) \
    bitrev_table_16bit(16, (short *)(inout_buffer+512), seed_radix2_16bit_512); \
    fft_quad_frac32(512, inout_buffer, w_table_radix2_frac32_512); \
    fft_radix2_last_stage_frac32(512, inout_buffer, w_table_radix2_frac32_512)

#define fft_quad_frac32_2048(inout_buffer) \
    bitrev_table_16bit(32, (short *)(inout_buffer+2048), seed_radix2_16bit_2048); \
    fft_quad_frac32(2048, inout_buffer, w_table_radix2_frac32_2048); \
    fft_radix2_last_stage_frac32(2048, inout_buffer, w_table_radix2_frac32_2048)




/******************** real to complex FFTs macros *********************/
/* float */
#define fft_real2complex_float_128(inout_buffer) \
    fft_radix4_float_64(inout_buffer); \
    fft_real_split_float(128, inout_buffer, wa_table_float_128, wb_table_float_128)

#define fft_real2complex_float_512(inout_buffer) \
    fft_radix4_float_256(inout_buffer); \
    fft_real_split_float(512, inout_buffer, wa_table_float_512, wb_table_float_512)

#define fft_real2complex_float_2048(inout_buffer) \
    fft_radix4_float_1024(inout_buffer); \
    fft_real_split_float(2048, inout_buffer, wa_table_float_2048, wb_table_float_2048)

#define fft_real2complex_float_256(inout_buffer) \
    fft_quad_float_128(inout_buffer); \
    fft_real_split_float(256, inout_buffer, wa_table_float_256, wb_table_float_256)

#define fft_real2complex_float_1024(inout_buffer) \
    fft_quad_float_512(inout_buffer); \
    fft_real_split_float(1024, inout_buffer, wa_table_float_1024, wb_table_float_1024)

#define fft_real2complex_float_4096(inout_buffer) \
    fft_quad_float_2048(inout_buffer); \
    fft_real_split_float(4096, inout_buffer, wa_table_float_4096, wb_table_float_4096)




/* frac32 */
#define fft_real2complex_frac32_128(inout_buffer) \
    fft_radix4_frac32_64(inout_buffer); \
    fft_real_split_frac32(128, inout_buffer, wa_table_frac32_128, wb_table_frac32_128)

#define fft_real2complex_frac32_512(inout_buffer) \
    fft_radix4_frac32_256(inout_buffer); \
    fft_real_split_frac32(512, inout_buffer, wa_table_frac32_512, wb_table_frac32_512)

#define fft_real2complex_frac32_2048(inout_buffer) \
    fft_radix4_frac32_1024(inout_buffer); \
    fft_real_split_frac32(2048, inout_buffer, wa_table_frac32_2048, wb_table_frac32_2048)

#define fft_real2complex_frac32_256(inout_buffer) \
    fft_quad_frac32_128(inout_buffer); \
    fft_real_split_frac32(256, inout_buffer, wa_table_frac32_256, wb_table_frac32_256)

#define fft_real2complex_frac32_1024(inout_buffer) \
    fft_quad_frac32_512(inout_buffer); \
    fft_real_split_frac32(1024, inout_buffer, wa_table_frac32_1024, wb_table_frac32_1024)

#define fft_real2complex_frac32_4096(inout_buffer) \
    fft_quad_frac32_2048(inout_buffer); \
    fft_real_split_frac32(4096, inout_buffer, wa_table_frac32_4096, wb_table_frac32_4096)




#endif
